unit bkPageControl;

{$R-}

interface

uses Messages, Windows, SysUtils, CommCtrl, Classes,
     Controls, Forms, Menus, Graphics, StdCtrls, RichEdit,
     ToolWin, ImgList, ExtCtrls, ComCtrls, {DsgnIntf,} Buttons,
     Dialogs, Grids;

type
  TbkPageControl = class;

  TbkTabSheet = class(TWinControl)
  private
    FImageIndex: Integer;
    FPageControl: TbkPageControl;
    FTabVisible: Boolean;
    FTabShowing: Boolean;
    FOnHide: TNotifyEvent;
    FOnShow: TNotifyEvent;
    FTabColor: TColor;
    FTabFontColor: TColor;
    function GetPageIndex: Integer;
    function GetTabIndex: Integer;
    procedure SetTabColor(Value: TColor);
    procedure SetTabFontColor(Value: TColor);
    procedure SetImageIndex(Value: Integer);
    procedure SeTbkPageControl(APageControl: TbkPageControl);
    procedure SetPageIndex(Value: Integer);
    procedure SetTabShowing(Value: Boolean);
    procedure SetTabVisible(Value: Boolean);
    procedure UpdateTabShowing;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMShowingChanged(var Message: TMessage); message CM_SHOWINGCHANGED;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DoHide; dynamic;
    procedure DoShow; dynamic;
    procedure ReadState(Reader: TReader); override;
  public
// Code added by BK to add labels and a stringgrid to the TabSheet.  15 Mar 01
    PnTop: TPanel;
    LblProcessValue: TLabel;
    LblProcessName: TLabel;
    LblProcessUnits: TLabel;
    SgParams: TStringGrid;
// End of added code
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property PageControl: TbkPageControl read FPageControl write SeTbkPageControl;
    property TabIndex: Integer read GetTabIndex;
  published
    property BorderWidth;
    property Caption;
    property DragMode;
    property Enabled;
    property Font;
    property Height stored False;
    property ImageIndex: Integer read FImageIndex write SetImageIndex default 0;
    property Left stored False;
    property Constraints;
    property PageIndex: Integer read GetPageIndex write SetPageIndex stored False;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabColor: TColor read FTabColor write SetTabColor;
    property TabFontColor: TColor read FTabFontColor write SetTabFontColor;
    property TabVisible: Boolean read FTabVisible write SetTabVisible default True;
    property Top stored False;
    property Visible stored False;
    property Width stored False;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnHide: TNotifyEvent read FOnHide write FOnHide;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnShow: TNotifyEvent read FOnShow write FOnShow;
    property OnStartDrag;
  end;

  TbkPageControl = class(TCustomTabControl)
  private
    FPages: TList;
    FActivePage: TbkTabSheet;
    FNewDockSheet: TbkTabSheet;
    FUndockingPage: TbkTabSheet;
    procedure ChangeActivePage(Page: TbkTabSheet);
    procedure DeleteTab(Page: TbkTabSheet; Index: Integer);
    function GetDockClientFromMousePos(MousePos: TPoint): TControl;
    function GetPage(Index: Integer): TbkTabSheet;
    function GetPageCount: Integer;
    procedure InsertPage(Page: TbkTabSheet);
    procedure InsertTab(Page: TbkTabSheet);
    procedure MoveTab(CurIndex, NewIndex: Integer);
    procedure RemovePage(Page: TbkTabSheet);
    procedure SetActivePage(Page: TbkTabSheet);
    procedure UpdateTab(Page: TbkTabSheet);
    procedure UpdateActivePage;
    procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
    procedure CMDialogKey(var Message: TCMDialogKey); message CM_DIALOGKEY;
    procedure CMDockClient(var Message: TCMDockClient); message CM_DOCKCLIENT;
    procedure CMDockNotification(var Message: TCMDockNotification); message CM_DOCKNOTIFICATION;
    procedure CMUnDockClient(var Message: TCMUnDockClient); message CM_UNDOCKCLIENT;
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
  protected
    function CanShowTab(TabIndex: Integer): Boolean; override;
    procedure Change; override;
    procedure DoAddDockClient(Client: TControl; const ARect: TRect); override;
    procedure DockOver(Source: TDragDockObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean); override;
    procedure DoRemoveDockClient(Client: TControl); override;
    procedure DrawTab(TabIndex: Integer; const Rect: TRect; Active: Boolean); override;
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
    function GetImageIndex(TabIndex: Integer): Integer; override;
    function GetPageFromDockClient(Client: TControl): TbkTabSheet;
    procedure GetSiteInfo(Client: TControl; var InfluenceRect: TRect;
      MousePos: TPoint; var CanDock: Boolean); override;
    procedure SetChildOrder(Child: TComponent; Order: Integer); override;
    procedure ShowControl(AControl: TControl); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function FindNextPage(CurPage: TbkTabSheet;
      GoForward, CheckTabVisible: Boolean): TbkTabSheet;
    procedure SelectNextPage(GoForward: Boolean);
    property PageCount: Integer read GetPageCount;
    property Pages[Index: Integer]: TbkTabSheet read GetPage;
  published
    property ActivePage: TbkTabSheet read FActivePage write SetActivePage;
    property Align;
    property Anchors;
    property BiDiMode;
    property Constraints;
    property DockSite;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property HotTrack;
    property Images;
    property MultiLine;
    property OwnerDraw;
    property ParentBiDiMode;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property RaggedRight;
    property ScrollOpposite;
    property ShowHint;
    property Style;
    property TabHeight;
    property TabOrder;
    property TabPosition;
    property TabStop;
    property TabWidth;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnDockDrop;
    property OnDockOver;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawTab;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetImageIndex;
    property OnGetSiteInfo;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnStartDock;
    property OnStartDrag;
    property OnUnDock;
  end;

{  TbkPageControlEditor = class(TDefaultEditor)
  protected
  public
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
  end;          }

implementation

{$R *.res}

uses ComStrs, stypes;

{ TbkTabSheet }

constructor TbkTabSheet.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  TabColor := clBtnFace;
  TabFontColor := clBlack;
  Align := alClient;
  ControlStyle := ControlStyle + [csAcceptsControls, csNoDesignVisible];
  Visible := False;
  FTabVisible := True;
// Code added by BK to add labels and a stringgrid to the TabSheet.  16 Mar 01
  PnTop := TPanel.Create(Self);
  with PnTop do
   begin
     Parent := Self;
     Caption := '';
     Align := alTop
   end;
  LblProcessValue := TLabel.Create(Self);
  with LblProcessValue do
    begin
      Parent:=PnTop;
      Caption := '';
      Top := 14;
      Left := 182;
    end;
  LblProcessName := TLabel.Create(Self) ;
  with LblProcessName do
    begin
      Parent := PnTop;
      Caption := '';
      Top := 14;
      Left := 80;
    end;
  LblProcessUnits := TLabel.Create(Self) ;
  with LblProcessUnits do
    begin
      Parent := PnTop;
      Caption := '';
      Top := 14;
      Left := 310;
    end;
  SgParams := TStringGrid.Create(Self) ;
  with SgParams do
    begin
      Parent := Self;
      Align := AlClient;
      Height := font.height;
      ColCount := 4;
      DefaultColWidth := round(0.7*stringlength*font.size);
 //     ColWidths[0] := DefaultColWidth div 2;
      Cells[0,0] := 'Parameter';
      Cells[1,0] := 'Parameter Name';
      Cells[2,0] := 'Parameter Value';
      Cells[3,0] := 'Parameter Units';
    end;
// End of code added by BK.
end;

destructor TbkTabSheet.Destroy;
begin
  if FPageControl <> nil then
  begin
    if FPageControl.FUndockingPage = Self then FPageControl.FUndockingPage := nil;
    FPageControl.RemovePage(Self);
  end;
  inherited Destroy;
end;

procedure TbkTabSheet.DoHide;
begin
  if Assigned(FOnHide) then FOnHide(Self);
end;

procedure TbkTabSheet.DoShow;
begin
  if Assigned(FOnShow) then FOnShow(Self);
end;

function TbkTabSheet.GetPageIndex: Integer;
begin
  if FPageControl <> nil then
    Result := FPageControl.FPages.IndexOf(Self) else
    Result := -1;
end;

function TbkTabSheet.GetTabIndex: Integer;
var
  I: Integer;
begin
  Result := 0;
  if not FTabShowing then Dec(Result) else
    for I := 0 to PageIndex - 1 do
      if TbkTabSheet(FPageControl.FPages[I]).FTabShowing then
        Inc(Result);
end;

procedure TbkTabSheet.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params.WindowClass do
    style := style and not (CS_HREDRAW or CS_VREDRAW);
end;

procedure TbkTabSheet.ReadState(Reader: TReader);
begin
  inherited ReadState(Reader);
  if Reader.Parent is TbkPageControl then
    PageControl := TbkPageControl(Reader.Parent);
end;

procedure TbkTabSheet.SetImageIndex(Value: Integer);
begin
  if FImageIndex <> Value then
  begin
    FImageIndex := Value;
    if FTabShowing then FPageControl.UpdateTab(Self);
  end;
end;

procedure TbkTabSheet.SeTbkPageControl(APageControl: TbkPageControl);
begin
  if FPageControl <> APageControl then
  begin
    if FPageControl <> nil then FPageControl.RemovePage(Self);
    Parent := APageControl;
    if APageControl <> nil then APageControl.InsertPage(Self);
  end;
end;

procedure TbkTabSheet.SetPageIndex(Value: Integer);
var
  I, MaxPageIndex: Integer;
begin
  if FPageControl <> nil then
  begin
    MaxPageIndex := FPageControl.FPages.Count - 1;
    if Value > MaxPageIndex then
      raise EListError.CreateFmt(SPageIndexError, [Value, MaxPageIndex]);
    I := TabIndex;
    FPageControl.FPages.Move(PageIndex, Value);
    if I >= 0 then FPageControl.MoveTab(I, TabIndex);
  end;
end;

procedure TbkTabSheet.SetTabShowing(Value: Boolean);
var
  Index: Integer;
begin
  if FTabShowing <> Value then
    if Value then
    begin
      FTabShowing := True;
      FPageControl.InsertTab(Self);
    end else
    begin
      Index := TabIndex;
      FTabShowing := False;
      FPageControl.DeleteTab(Self, Index);
    end;
end;

procedure TbkTabSheet.SetTabVisible(Value: Boolean);
begin
  if FTabVisible <> Value then
  begin
    FTabVisible := Value;
    UpdateTabShowing;
  end;
end;

procedure TbkTabSheet.UpdateTabShowing;
begin
  SetTabShowing((FPageControl <> nil) and
    (FTabVisible or (csDesigning in ComponentState)));
end;

procedure TbkTabSheet.CMTextChanged(var Message: TMessage);
begin
  if FTabShowing then FPageControl.UpdateTab(Self);
end;

procedure TbkTabSheet.CMShowingChanged(var Message: TMessage);
begin
  inherited;
  if Showing then
  begin
    try
      DoShow
    except
      Application.HandleException(Self);
    end;
  end else if not Showing then
  begin
    try
      DoHide;
    except
      Application.HandleException(Self);
    end;
  end;
end;

procedure TbkTabSheet.SetTabColor(Value: TColor);
begin
  FTabColor := Value;
end;

procedure TbkTabSheet.SetTabFontColor(Value: TColor);
begin
  FTabFontColor := Value;
end;

{ TbkPageControl }

constructor TbkPageControl.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  OwnerDraw := True;
  ControlStyle := [csDoubleClicks, csOpaque];
  FPages := TList.Create;
end;

destructor TbkPageControl.Destroy;
var
  I: Integer;
begin
  for I := 0 to FPages.Count - 1 do TbkTabSheet(FPages[I]).FPageControl := nil;
  FPages.Free;
  inherited Destroy;
end;

function TbkPageControl.CanShowTab(TabIndex: Integer): Boolean;
begin
  Result := TbkTabSheet(FPages[TabIndex]).Enabled;
end;

procedure TbkPageControl.Change;
var
  Form: TCustomForm;
begin
  UpdateActivePage;
  if csDesigning in ComponentState then
  begin
    Form := GetParentForm(Self);
    if (Form <> nil) and (Form.Designer <> nil) then Form.Designer.Modified;
  end;
  inherited Change;
end;

procedure TbkPageControl.ChangeActivePage(Page: TbkTabSheet);
var
  ParentForm: TCustomForm;
begin
  if FActivePage <> Page then
  begin
    ParentForm := GetParentForm(Self);
    if (ParentForm <> nil) and (FActivePage <> nil) and
      FActivePage.ContainsControl(ParentForm.ActiveControl) then
    begin
      ParentForm.ActiveControl := FActivePage;
      if ParentForm.ActiveControl <> FActivePage then
      begin
        TabIndex := FActivePage.TabIndex;
        Exit;
      end;
    end;
    if Page <> nil then
    begin
      Page.BringToFront;
      Page.Visible := True;
      if (ParentForm <> nil) and (FActivePage <> nil) and
        (ParentForm.ActiveControl = FActivePage) then
        if Page.CanFocus then
          ParentForm.ActiveControl := Page else
          ParentForm.ActiveControl := Self;
    end;
    if FActivePage <> nil then FActivePage.Visible := False;
    FActivePage := Page;
    if (ParentForm <> nil) and (FActivePage <> nil) and
      (ParentForm.ActiveControl = FActivePage) then
      FActivePage.SelectFirst;
  end;
end;

procedure TbkPageControl.DeleteTab(Page: TbkTabSheet; Index: Integer);
var
  UpdateIndex: Boolean;
begin
  UpdateIndex := Page = ActivePage;
  Tabs.Delete(Index);
  if UpdateIndex then
  begin
    if Index >= Tabs.Count then
      Index := Tabs.Count - 1; 
    TabIndex := Index;
  end;
  UpdateActivePage;
end;

procedure TbkPageControl.DoAddDockClient(Client: TControl; const ARect: TRect);
begin
  if FNewDockSheet <> nil then Client.Parent := FNewDockSheet;
end;

procedure TbkPageControl.DockOver(Source: TDragDockObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
var
  R: TRect;
begin
  GetWindowRect(Handle, R);
  Source.DockRect := R;
  DoDockOver(Source, X, Y, State, Accept);
end;

procedure TbkPageControl.DoRemoveDockClient(Client: TControl);
begin
  if (FUndockingPage <> nil) and not (csDestroying in ComponentState) then
  begin
    SelectNextPage(True);
    FUndockingPage.Free;
    FUndockingPage := nil;
  end;
end;

function TbkPageControl.FindNextPage(CurPage: TbkTabSheet;
  GoForward, CheckTabVisible: Boolean): TbkTabSheet;
var
  I, StartIndex: Integer;
begin
  if FPages.Count <> 0 then
  begin
    StartIndex := FPages.IndexOf(CurPage);
    if StartIndex = -1 then
      if GoForward then StartIndex := FPages.Count - 1 else StartIndex := 0;
    I := StartIndex;
    repeat
      if GoForward then
      begin
        Inc(I);
        if I = FPages.Count then I := 0;
      end else
      begin
        if I = 0 then I := FPages.Count;
        Dec(I);
      end;
      Result := FPages[I];
      if not CheckTabVisible or Result.TabVisible then Exit;
    until I = StartIndex;
  end;
  Result := nil;
end;

procedure TbkPageControl.GetChildren(Proc: TGetChildProc; Root: TComponent);
var
  I: Integer;
begin
  for I := 0 to FPages.Count - 1 do Proc(TComponent(FPages[I]));
end;

function TbkPageControl.GetImageIndex(TabIndex: Integer): Integer;
begin
{  if Assigned(FOnGetImageIndex) then
    Result := inherited GetImageIndex(TabIndex) else}
    Result := GetPage(TabIndex).ImageIndex;
end;

function TbkPageControl.GetPageFromDockClient(Client: TControl): TbkTabSheet;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to PageCount - 1 do
  begin
    if (Client.Parent = Pages[I]) and (Client.HostDockSite = Self) then
    begin
      Result := Pages[I];
      Exit;
    end;
  end;
end;

function TbkPageControl.GetPage(Index: Integer): TbkTabSheet;
begin
  Result := FPages[Index];
end;

function TbkPageControl.GetPageCount: Integer;
begin
  Result := FPages.Count;
end;

procedure TbkPageControl.GetSiteInfo(Client: TControl; var InfluenceRect: TRect;
  MousePos: TPoint; var CanDock: Boolean);
begin
  CanDock := GetPageFromDockClient(Client) = nil;
  inherited GetSiteInfo(Client, InfluenceRect, MousePos, CanDock);
end;

procedure TbkPageControl.InsertPage(Page: TbkTabSheet);
begin
  FPages.Add(Page);
  Page.FPageControl := Self;
  Page.UpdateTabShowing;
end;

procedure TbkPageControl.InsertTab(Page: TbkTabSheet);
begin
  Tabs.InsertObject(Page.TabIndex, Page.Caption, Page);
  UpdateActivePage;
end;

procedure TbkPageControl.MoveTab(CurIndex, NewIndex: Integer);
begin
  Tabs.Move(CurIndex, NewIndex);
end;

procedure TbkPageControl.RemovePage(Page: TbkTabSheet);
var
  NextSheet: TbkTabSheet;
begin
  NextSheet := FindNextPage(Page, True, not (csDesigning in ComponentState));
  if NextSheet = Page then NextSheet := nil;
  Page.SetTabShowing(False);
  Page.FPageControl := nil;
  FPages.Remove(Page);
  SetActivePage(NextSheet);
end;

procedure TbkPageControl.SelectNextPage(GoForward: Boolean);
var
  Page: TbkTabSheet;
begin
  Page := FindNextPage(ActivePage, GoForward, True);
  if (Page <> nil) and (Page <> ActivePage) and CanChange then
  begin
    TabIndex := Page.TabIndex;
    Change;
  end;
end;

procedure TbkPageControl.SetActivePage(Page: TbkTabSheet);
begin
  if (Page <> nil) and (Page.PageControl <> Self) then Exit;
  ChangeActivePage(Page);
  if Page = nil then
    TabIndex := -1
  else if Page = FActivePage then
    TabIndex := Page.TabIndex;
end;

procedure TbkPageControl.SetChildOrder(Child: TComponent; Order: Integer);
begin
  TbkTabSheet(Child).PageIndex := Order;
end;

procedure TbkPageControl.ShowControl(AControl: TControl);
begin
  if (AControl is TbkTabSheet) and (TbkTabSheet(AControl).PageControl = Self) then
    SetActivePage(TbkTabSheet(AControl));
  inherited ShowControl(AControl);
end;

procedure TbkPageControl.UpdateTab(Page: TbkTabSheet);
begin
  Tabs[Page.TabIndex] := Page.Caption;
end;

procedure TbkPageControl.UpdateActivePage;
begin
  if TabIndex >= 0 then
    SetActivePage(TbkTabSheet(Tabs.Objects[TabIndex]))
  else
    SetActivePage(nil);
end;

procedure TbkPageControl.CMDesignHitTest(var Message: TCMDesignHitTest);
var
  HitIndex: Integer;
  HitTestInfo: TTCHitTestInfo;
begin
  HitTestInfo.pt := SmallPointToPoint(Message.Pos);
  HitIndex := SendMessage(Handle, TCM_HITTEST, 0, Longint(@HitTestInfo));
  if (HitIndex >= 0) and (HitIndex <> TabIndex) then Message.Result := 1;
end;

procedure TbkPageControl.CMDialogKey(var Message: TCMDialogKey);
begin
  if (Focused or Windows.IsChild(Handle, Windows.GetFocus)) and
    (Message.CharCode = VK_TAB) and (GetKeyState(VK_CONTROL) < 0) then
  begin
    SelectNextPage(GetKeyState(VK_SHIFT) >= 0);
    Message.Result := 1;
  end else
    inherited;
end;

procedure TbkPageControl.CMDockClient(var Message: TCMDockClient);
var
  IsVisible: Boolean;
  DockCtl: TControl;
begin
  Message.Result := 0;
  FNewDockSheet := TbkTabSheet.Create(Self);
  try
    try
      DockCtl := Message.DockSource.Control;
      if DockCtl is TCustomForm then
        FNewDockSheet.Caption := TCustomForm(DockCtl).Caption;
      FNewDockSheet.PageControl := Self;
      DockCtl.Dock(Self, Message.DockSource.DockRect);
    except
      FNewDockSheet.Free;
      raise;
    end;
    IsVisible := DockCtl.Visible;
    FNewDockSheet.TabVisible := IsVisible;
    if IsVisible then ActivePage := FNewDockSheet;
    DockCtl.Align := alClient;
  finally
    FNewDockSheet := nil;
  end;
end;

procedure TbkPageControl.CMDockNotification(var Message: TCMDockNotification);
var
  I: Integer;
  S: string;
  Page: TbkTabSheet;
begin
  Page := GetPageFromDockClient(Message.Client);
  if Page <> nil then
    case Message.NotifyRec.ClientMsg of
      WM_SETTEXT:
        begin
          S := PChar(Message.NotifyRec.MsgLParam);
          { Search for first CR/LF and end string there }
          for I := 1 to Length(S) do
            if S[I] in [#13, #10] then
            begin
              SetLength(S, I - 1);
              Break;
            end;
          Page.Caption := S;
        end;
      CM_VISIBLECHANGED:
        with Page do
        begin
          Visible := Boolean(Message.NotifyRec.MsgWParam);
          TabVisible := Boolean(Message.NotifyRec.MsgWParam);;
        end;
    end;
  inherited;
end;

procedure TbkPageControl.CMUnDockClient(var Message: TCMUnDockClient);
var
  Page: TbkTabSheet;
begin
  Message.Result := 0;
  Page := GetPageFromDockClient(Message.Client);
  if Page <> nil then
  begin
    FUndockingPage := Page;
    Message.Client.Align := alNone;
  end;
end;

function TbkPageControl.GetDockClientFromMousePos(MousePos: TPoint): TControl;
var
  HitIndex: Integer;
  HitTestInfo: TTCHitTestInfo;
  Page: TbkTabSheet;
begin
  Result := nil;
  if DockSite then
  begin
    HitTestInfo.pt := MousePos;
    HitIndex := SendMessage(Handle, TCM_HITTEST, 0, Longint(@HitTestInfo));
    if HitIndex >= 0 then
    begin
      Page := Pages[HitIndex];
      if not Page.TabVisible then Page := FindNextPage(Page, True, True);
      if (Page <> nil) and (Page.ControlCount > 0) then
      begin
        Result := Page.Controls[0];
        if Result.HostDockSite <> Self then Result := nil;
      end;
    end;
  end;
end;

procedure TbkPageControl.WMLButtonDown(var Message: TWMLButtonDown);
var
  DockCtl: TControl;
begin
  inherited;
  DockCtl := GetDockClientFromMousePos(SmallPointToPoint(Message.Pos));
  if DockCtl <> nil then DockCtl.BeginDrag(False);
end;

procedure TbkPageControl.WMLButtonDblClk(var Message: TWMLButtonDblClk);
var
  DockCtl: TControl;
begin
  inherited;
  DockCtl := GetDockClientFromMousePos(SmallPointToPoint(Message.Pos));
  if DockCtl <> nil then DockCtl.ManualDock(nil, nil, alNone);
end;

procedure TbkPageControl.DrawTab(TabIndex: Integer; const Rect: TRect;
  Active: Boolean);
begin
  inherited;
  with Canvas do
  begin
    Brush.Color := Pages[TabIndex].TabColor;
    FillRect(Rect);
    Font.Color := Pages[TabIndex].TabFontColor;
    TextRect(Rect,Rect.Left + 4, Rect.Top + 4, Pages[TabIndex].Caption);
    DrawFocusRect(Rect);
  end;
end;

{ TbkPageControlEditor }

{procedure TbkPageControlEditor.ExecuteVerb(Index: Integer);
var
  NewPage: TbkTabSheet;
begin
  case Index of
    0:  begin  //  New Page
          NewPage := TbkTabSheet.Create(Designer.GetRoot);
          with NewPage do
          begin
            Parent := (Component as TbkPageControl);
            PageControl := (Component as TbkPageControl);
            Caption := 'ColorSheet' + IntToStr((Component as TbkPageControl).PageCount);
            Name := Caption;
          end;
        end;
    1:  begin  //  Delete Page
          with (Component as TbkPageControl) do
          begin
            NewPage := ActivePage;
            NewPage.PageControl := nil;
            NewPage.Free;
          end;
        end;
    2:  begin  //  Next Page
          (Component as TbkPageControl).FindNextPage((Component as TbkPageControl).ActivePage,True,False);
        end;
    3:  begin  //  Previous Page
          (Component as TbkPageControl).FindNextPage((Component as TbkPageControl).ActivePage,False,False);
        end;
  end;
  if Designer <> nil then Designer.Modified;
end;

function TbkPageControlEditor.GetVerb(Index: Integer): string;
begin
  case Index of
    0:  result := 'New Page';
    1:  result := 'Delete Page';
    2:  result := 'Next Page';
    3:  result := 'Previous Page';
  end;
end;

function TbkPageControlEditor.GetVerbCount: Integer;
begin
  result := 4;
end;
        }
end.

